<?php
/**
 * The Aheto
 *
 * @since      1.0.0
 * @package    Aheto
 * @subpackage Aheto
 * @author     UPQODE <info@upqode.com>
 */

namespace Aheto;

use Aheto\Helper;
use Aheto\Settings;
use Aheto\Post_Types;
use Aheto\Json_Center;
use Aheto\Traits\Hooker;
use Aheto\Admin\Admin;
use Aheto\Admin\Mega_Menu;
use Aheto\Builder_Loader;
use Aheto\Frontend\Frontend;
use Aheto\Notification_Center;
use Aheto\Template_Kit\API;

defined( 'ABSPATH' ) || exit;

/**
 * Main Aheto Class.
 */
class Aheto {

	use Hooker;

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	public $version = '0.9.0';

	/**
	 * Plugin database version.
	 *
	 * @var string
	 */
	public $db_version = '1';

	/**
	 * The single instance of the class.
	 *
	 * @var Aheto
	 */
	protected static $_instance = null;

	/**
	 * Possible error message.
	 *
	 * @var null|WP_Error
	 */
	protected $error = null;

	/**
	 * Halt plugin loading.
	 *
	 * @var boolean
	 */
	private $is_critical = false;

	/**
	 * Minimum WordPress version required by the plugin.
	 *
	 * @var string
	 */
	public $wordpress_version = '3.8';

	/**
	 * Minimum PHP version required by the plugin.
	 *
	 * @var string
	 */
	public $php_version = '5.6';

	/**
	 * Factory.
	 *
	 * @var array
	 */
	private $container = [];

	/**
	 * Plugin url.
	 *
	 * @var string
	 */
	private $plugin_url = null;

	/**
	 * Plugin path.
	 *
	 * @var string
	 */
	private $plugin_dir = null;

	/**
	 * Cloning is forbidden.
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning is forbidden.', 'aheto' ), $this->version );
	}

	/**
	 * Unserializing instances of this class is forbidden.
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing instances of this class is forbidden.', 'aheto' ), $this->version );
	}

	/**
	 * Magic getter to bypass referencing plugin.
	 *
	 * @param  string $prop Key to get.
	 * @return mixed
	 */
	public function __get( $prop ) {

		if ( array_key_exists( $prop, $this->container ) ) {
			return $this->container[ $prop ];
		}

		return $this->$prop;
	}

	/**
	 * Magic setter to bypass referencing plugin.
	 *
	 * @param mixed $prop  Key to set.
	 * @param mixed $value Value to set.
	 */
	public function __set( $prop, $value ) {

		if ( property_exists( $this, $prop ) ) {
			$this->$prop = $value;
		}

		$this->container[ $prop ] = $value;
	}

	/**
	 * Magic isset to bypass referencing plugin.
	 *
	 * @param mixed $prop Property to check.
	 * @return mixed
	 */
	public function __isset( $prop ) {
		return isset( $this->{$prop} ) || isset( $this->container[ $prop ] );
	}

	/**
	 * Main Aheto instance.
	 *
	 * Ensure only one instance is loaded or can be loaded.
	 *
	 * @see aheto()
	 * @return Aheto
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) && ! ( self::$_instance instanceof Aheto ) ) {
			self::$_instance = new Aheto();
			self::$_instance->setup();

		}

		return self::$_instance;
	}

	/**
	 * Aheto constructor.
	 */
	private function __construct() {}

	/**
	 * Setup the plugin.
	 *
	 * Sets up all the appropriate hooks and actions within our plugin.
	 */
	private function setup() {

		if ( $this->check_requirements() ) {
			return;
		}

		$this->hooks();

		// Loaded action.
		$this->do_action( 'loaded' );
	}

	/**
	 * Hook into actions and filters.
	 */
	private function hooks() {
		register_activation_hook( AHETO_FILE, [ 'Aheto\\Installer', 'install' ] );
		register_deactivation_hook( AHETO_FILE, 'flush_rewrite_rules' );

		$this->action( 'init', 'instantiate', 0 );
		$this->action( 'init', 'load_plugin_textdomain' );

		$this->action( 'wp_ajax_likes_ajax',        'likes_function_callback', 10 );
		$this->action( 'wp_ajax_nopriv_likes_ajax', 'likes_function_callback', 10 );

		$this->filter( 'plugin_row_meta', 'plugin_row_meta', 10, 2 );
		$this->filter( 'plugin_action_links_' . plugin_basename( AHETO_FILE ), 'plugin_action_links' );
	}

	/**
	 * Instantiate classes.
	 *
	 * @return void
	 */
	public function instantiate() {
		$this->json         = new Json_Center;
		$this->settings     = new Settings;
		$this->notification = new Notification_Center;

		// Just Init.
		new Post_Types;
		new Builder_Loader;
		new Page_Templates;
		new Mega_Menu;

		$this->settings->add_option( 'aheto-general-settings', 'general' );

		$this->init_css_generator();

		// Admin Only.
		if ( is_admin() ) {
			$this->admin = new Admin;
		}

		// Frontend Only.
		if ( ! is_admin() || 'elementor' === Helper::param_get( 'action' ) || 'elementor_ajax' === Helper::param_post( 'action' ) ) {
			$this->frontend = new Frontend;
		}
	}

	/**
	 * Init css generator
	 */
	private function init_css_generator() {
		$hash = [
			'elementor'       => '\\Aheto\\CSS\\Generator\\Elementor',
			'visual-composer' => '\\Aheto\\CSS\\Generator\\Visual_Composer',
		];

		$builder = Helper::get_settings( 'general.builder' );
		if ( isset( $hash[ $builder ] ) ) {
			$this->container['css_generator'] = new $hash[ $builder ];
		}
	}

	/**
	 * Load Localisation files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *     - WP_LANG_DIR/aheto/aheto-LOCALE.mo
	 *     - WP_LANG_DIR/plugins/aheto-LOCALE.mo
	 */
	public function load_plugin_textdomain() {
		$locale = is_admin() && function_exists( 'get_user_locale' ) ? get_user_locale() : get_locale();
		$locale = apply_filters( 'plugin_locale', $locale, 'aheto' );

		unload_textdomain( 'aheto' );
		load_textdomain( 'aheto', WP_LANG_DIR . '/aheto/aheto-' . $locale . '.mo' );
		load_plugin_textdomain( 'aheto', false, $this->plugin_dir() . '/languages/' );
	}

	/**
	 * Show action links on the plugin screen.
	 *
	 * @param  mixed $links Plugin Action links.
	 * @return array
	 */
	public function plugin_action_links( $links ) {
		$more = [ '<a href="' . Helper::get_admin_url( 'general-settings' ) . '">' . esc_html__( 'Settings', 'aheto' ) . '</a>' ];

		if ( Helper::is_setup_wizard() ) {
			$more[] = '<a href="' . Helper::get_admin_url( 'setup' ) . '">' . esc_html__( 'Setup Wizard', 'aheto' ) . '</a>';
		}

		return array_merge( $links, $more );
	}

	/**
	 * Function callback for likes Ajax
	 *
	 */
	public function likes_function_callback() {

		$user = wp_get_current_user();
		$post = $_POST['id_p'];

		$array = get_post_meta( $post, 'aheto_post_likes', true );
		if ( empty( $array ) ) {
			$array = array();
		}

		if ( $array[$user->ID] && ( $array[$user->ID] == 1 ) ) {
			unset( $array[$user->ID] );
		}
		else {
			$array[$user->ID] = 1;
		}

		update_post_meta( $post, 'aheto_post_likes', $array );

		wp_send_json( [ 'success' => 'true', 'message' => 'Done', 'result_count' => count($array) ], 200 );
	}

	/**
	 * Show row meta on the plugin screen.
	 *
	 * @param  mixed $links Plugin Row Meta.
	 * @param  mixed $file  Plugin Base file.
	 * @return array
	 */
	public function plugin_row_meta( $links, $file ) {

		if ( plugin_basename( AHETO_FILE ) !== $file ) {
			return $links;
		}

		return array_merge( $links, [
			'<a href="' . Helper::get_admin_url( 'about' ) . '">' . esc_html__( 'Getting Started', 'aheto' ) . '</a>',
			'<a href="' . esc_url( '#doc_url' ) . '">' . esc_html__( 'Documentation', 'aheto' ) . '</a>',
		] );
	}

	/**
	 * Check that the WordPress and PHP setup meets the plugin requirements.
	 *
	 * @return boolean
	 */
	private function check_requirements() {

		$this->is_wp_enough();
		$this->is_php_enough();

		return $this->is_critical;
	}

	/**
	 * Check if WordPress version is enough to run this plugin.
	 */
	public function is_wp_enough() {

		if ( version_compare( get_bloginfo( 'version' ), $this->wordpress_version, '<' ) ) {
			$this->add_error(
			/* translators: WordPress Version */
				sprintf( aheto()->plugin_name() . esc_html__( ' requires WordPress version %s or above. Please update WordPress to run this plugin.', 'aheto' ), $this->wordpress_version )
			);
			$this->is_critical = true;
		}
	}

	/**
	 * Check if PHP version is enough to run this plugin.
	 */
	public function is_php_enough() {

		if ( version_compare( phpversion(), $this->php_version, '<' ) ) {
			$this->add_error(
			/* translators: PHP Version */
				sprintf( aheto()->plugin_name() . esc_html__( ' requires PHP version %s or above. Please update PHP to run this plugin.', 'aheto' ), $this->php_version )
			);
			$this->is_critical = true;
		}
	}

	/**
	 * Get the plugin dir.
	 *
	 * @return string
	 */
	public function plugin_dir() {

		if ( is_null( $this->plugin_dir ) ) {
			$this->plugin_dir = untrailingslashit( plugin_dir_path( AHETO_FILE ) ) . '/';
		}

		return $this->plugin_dir;
	}

	/**
	 * Get the plugin url.
	 *
	 * @return string
	 */
	public function plugin_url() {

		if ( is_null( $this->plugin_url ) ) {
			$this->plugin_url = untrailingslashit( plugin_dir_url( AHETO_FILE ) ) . '/';
		}

		return $this->plugin_url;
	}

	/**
	 * Get the plugin name.
	 *
	 * @return string
	 */
	public function plugin_name() {

		$plugin_name = 'Aheto';

		return apply_filters( "aheto_plugin_name", $plugin_name );
	}

	/**
	 * Get the plugin icon.
	 *
	 * @return string
	 */


	public function plugin_icon() {

		$plugin_icon =  AHETO_URL . '/assets/admin/img/logos/small-logo.png';

		return apply_filters( "aheto_plugin_icon", $plugin_icon );
	}


	/**
	 * Get the plugin desktop logo.
	 *
	 * @return string
	 */


	public function plugin_dashboard_desktop_logo() {

		$plugin_dashboard_desktop_logo =  aheto()->assets() . 'admin/img/logos/desc-logo.png';

		return apply_filters( "aheto_plugin_dashboard_desktop_logo", $plugin_dashboard_desktop_logo );
	}



	/**
	 * Get the plugin desktop logo.
	 *
	 * @return string
	 */


	public function plugin_dashboard_mobile_logo() {

		$plugin_dashboard_mobile_logo =  aheto()->assets() . 'admin/img/logos/mob-logo.png';

		return apply_filters( "aheto_plugin_dashboard_mobile_logo", $plugin_dashboard_mobile_logo );
	}



	/**
	 * Set the plugin dashboard pages.
	 *
	 * @return string
	 */


	public function plugin_dashboard_pages() {

		$api = new API;
		$templates = $api->get_templates();
		$tags = array();

		$templates = is_array($templates) ? $templates : explode(",", $templates);

		foreach ($templates as $template){
			$tag = isset($template['tags']['classes'][0]) && !empty($template['tags']['classes'][0]) ? $template['tags']['classes'][0] : '';

			if(!empty($tag) && !in_array($tag, $tags)){
				$tags[] = $tag;
			}
		}

		$current_theme_dir = get_template_directory();
		$current_theme_dir = explode( '/themes/', $current_theme_dir );
		$current_theme_dir = $current_theme_dir[1];

		$plugin_dashboard_pages =  [
			'setup'            => [ aheto()->plugin_name() . esc_html__( ' Setup', 'aheto' ), 'ion-forward' ],
			'setting-up'       => [ esc_html__( 'Dashboard', 'aheto' ), 'ion-help-buoy' ],
			'general-settings' => [ esc_html__( 'Settings', 'aheto' ), 'ion-android-options' ],
			'theme-options'    => [ esc_html__( 'Theme Options', 'aheto' ), 'ion-android-settings' ],
			'skin-generator'   => [ esc_html__( 'Skins', 'aheto' ), 'ion-ios-color-wand' ],
			'templates'        => [ esc_html__( 'Template Kits', 'aheto' ), 'ion-grid' ],
			'import-export'    => [ esc_html__( 'Import &amp; Export', 'aheto' ), 'ion-android-sync' ],
		];

		if(!in_array($current_theme_dir, $tags)){
			unset($plugin_dashboard_pages['setup']);
		}

		return apply_filters( "aheto_plugin_dashboard_pages", $plugin_dashboard_pages );
	}


	/**
	 * Get assets url.
	 *
	 * @return string
	 */
	public function assets() {
		return $this->plugin_url() . 'assets/';
	}
}
